/** @file   sound.h
 * @brief   Declaration of Sound - class.
 * @version $Revision: 1.1.1.1 $
 * @date    $Date: 2006/01/21 23:02:38 $
 * @author  Tomi Lamminsaari
 */
 
#ifndef H_WWW_SOUNDPLAYER_H
#define H_WWW_SOUNDPLAYER_H

#include <vector>
#include <map>
#include <string>
#include "eng2d_dll.h"
#include "vec2d.h"
#include "soundsource.h"

namespace eng2d {





/** @class  Sound
 * @brief   Provides the interface to play sounds.
 * @author  Tomi Lamminsaari
 *
 * This class provides the services for playing sounds. You can play samples
 * or use positional sound sources for creating ambience audio.
 *
 * The positional sounds need to be updated regulary so you should keep calling
 * the @c update() - method which takes care of the sound control.
 */
class DLLIMPORT Sound
{
public:

  ///
  /// The installing and removing interface
  /// =====================================
  
  /** Number of voices we'd like to reserve for us. Default is 16. Set this
   * before calling the @c install() - method. See Allegro-documentation for
   * more information about the <code>void reserve_voices(int, int)</code>.
   */
  static int requestedVoices;
  
  /** Is the sounds on or off. @c true means that sounds are on. */
  static bool soundsOn;
  
  /** The master volume. NOT IMPLEMENTED ! */
  static int masterVolume;
  
  /** The default frequence. This is the frequence the playSample() - method
   * uses
   */
  static int defaultFrequency;
  
  /** The minimum replay delay. This defines the minimum number of calls
   * to @c update() - method before the same sample is replayed. Set -1 to
   * turn this feature off.
   *
   * Because the voice allocation is slow operation, it's a good thing to
   * prevent the same sample being replayed if it has been very recently
   * played already.
   */
  static int minimumReplayDelay;
  
  /** Installs the Sound Interface.
   * @return    Zero if successful or -1 if installing failed.
   */
  static int install();
  
  /** Releases the the Sound Interface.
   */
  static void remove();
  
  
  
  
  ///
  /// Static members
  /// ==============

protected:

  /** Vector that holds the pointers to all the samples. */
  static std::vector<SAMPLE*> m_samples;

  /** Each sample has also a name. We use this map to store the
   * samples by their names so that we can quickly find them.
   */
  static std::map<std::string, SAMPLE*> m_samplesByName;
  
  
public:

  ///
  /// The simple interface
  /// ====================
  
  /** Plays the sample. The sound will be placed center and played with
   * default volume 128.
   * @param     index             Index of the sample being plyed
   * @param     loop              Play the sample loopped or not.
   */
  static void playSample( int index, bool loop );
  
  /** Plays the sample.
   * @param     samplename        Name of the sample
   * @param     loop              Should it be loopped or not
   */
  static void playSample( const std::string& samplename, bool loop );
  
  
  /** Stops the index'th sample.
   * @param     index             Index of the sample that will be stopped.
   */
  static void stopSample( int index );
  
  /** Stops the playback of all samples.
   */
  static void stopAllSamples();
  
  
  
  ///
  /// The positional samples
  /// ======================
  
  
  /** Creates and playes an ambience sound source at given location.
   * The sample will be played loopped.
   * @param     pSource           Pointer to soundsource being added. This must
   *                              be dynamically created instance since we
   *                              delete it once the SoundSource has become
   *                              useless.
   */
  static void createSoundSource( eng2d::SoundSource* pSource );
  
  /** Removes the given soundsource
   * @param     pSource           Pointer to soundsource being removed.
   */
  static void removeSoundSource( eng2d::SoundSource* pSource );
  

  /** Updates the volumes of the samples. This method should be called
   * within regular intervals.
   * @param     rListener         Coordinates of the center of the screen.
   */
  static void update( const eng2d::Vec2D& rListener );
  
  
  
  /** Plays the sample so that the volume will be faded when the
   * distance grows
   * @param     index             Index of the sample being played
   * @param     distance          Distance of the sound
   */
  static void playDistanceSample( int index, float distance );
  
  /** Removes all the positional samples from the active sounds table.
   */
  static void clearSoundSources();
  
  /** Returns the number of positional samples there are.
   */
  static int numberOfSoundSources();
  
  /** Tells if the requested soundsource exists in the source table.
   * @param     pSS               Pointer to soundsource
   * @return    <code>true</code> if given soundsource exists
   */
  static bool soundSourceExists( eng2d::SoundSource* pSS );
  
  
  
  ///
  /// Default management
  /// ==================
  
  /** Loads the samples to memory.
   * @param     sfile             Name of the samplelist.
   * @return    0 if successfull. Nonzero if loading failed.
   */
  static int initSamples( const std::string& sfile );
  
  
  /** Removes the sound samples from memory.
   */
  static void flushSamples();
  
  
protected:

  ///
  /// Protected methods
  /// =================
  
  /** Reads the samplelist and loads the samples mentioned in that list.
   * @param     sfile             Name of the samplelist file
   * @return    Nonzero if fails. This method fails if we cannot open the
   *            samplelist file or if loading of at least one sample fails.
   */
  static int readSamples( const std::string& sfile );

  /** A counter that is updated in every call to @c update() - method. */
  static int updateCounter;
  
  /** Vector that holds all the positional samples added to the playtable. */
  static std::vector<eng2d::SoundSource*> m_soundsources;
  
  /** A vector that holds the times, when each of the samples are played
   * last time. To replay the same sample, the time difference must be
   * at least 2 update-rounds.
   */
  static std::vector< int > m_lastPlayTimes;
  
};


};  // end of namespace

#endif
